package com.sap.idm.vds.connector.samplefile.impl;

import java.io.*;
import java.io.RandomAccessFile;
import java.util.HashMap;


/**
 * This class is used for searching under a starting point. Filters can be used in order to filter results.
 * There are three different search types:
 * 		- base search: gives the attributes and values of one file or directory
 * 		- one level search: gives the list of files and directories of a directory
 * 		- subtree search: gives the complete subtree of files and directories under a directory
 * 
 * @author I054742
 *
 */
public class FileSearch {
	
	/* It will contain the search results */
	HashMap searchResult = null;
	/* It will contain a list of attributes when that is required */
	HashMap attrList = null;
	
	
	/**
	 * Constructor
	 */
	public FileSearch () {
		
		this.searchResult = new HashMap();
	}
	
	
	/**
	 * Returns the search result
	 * @return -> Returns a HashMap with the search results
	 */
	public HashMap getSearchResult () {
		
		return this.searchResult;
	}
	
	
	/**
	 * Method which is used to call to the correct search method depending of 'option'
	 * @param path - path of file or directory as starting point
	 * @param filter - filter for the results
	 * @param option - indicates which type of search must be done
	 * @param isDNDirectory - indicates if the expected type of file is a directory or not, according to the caller beliefs
	 * @return - An integer code indicating if the operation was done successfully or not.
	 * 			 In the case of error the corresponding error code is returned
	 */
	public int search (String path, String filter, int option, boolean isDNDirectory) {
		
		File file = new File(path);
		/* If the file does not exist the corresponding error code is returned */
		if (file.exists()==false) {
			return FileConstants.FILE_NOT_FOUND;
		}
		/* If the type of file does not correspond with the one specified by the DN 
		 * then the corresponding error code is returned */
		if (file.isDirectory()!=isDNDirectory) {
			return FileConstants.WRONG_DN_FILE_TYPE;
		}
		
		this.searchResult.clear();
		
		/* Makes the search according to the given option */
		if (option==FileConstants.BASE_SEARCH) return this.searchBase(path,filter);
		else if (option==FileConstants.ONE_LEVEL_SEARCH) return this.searchDeeper(path,filter,FileConstants.ONE_LEVEL_SEARCH);
		else if (option==FileConstants.SUBTREE_SEARCH) return this.searchDeeper(path,filter,FileConstants.SUBTREE_SEARCH);
		else return FileConstants.INVALID_OPTION;
	}
	
	
	/**
	 * Gives the attributes of a file or directory
	 * @param path -> Path of the file or directory
	 * @param filter -> Filter for the results. 
	 * 				   The filter must be a string following the next composition: attributeName filterString
	 * @return -> An integer code indicating if the operation was done successfully or not.
	 * 			  In the case of error the corresponding error code is returned
	 */
	private int searchBase (String path, String filter) {
		
		File file = new File(path);
		
		/* If the file does not exist then this is indicated by a error code */
		if (file.exists()==false) {
			return FileConstants.FILE_NOT_FOUND;
		}
		
		/* Makes the attribute lists of the file or directory */
		int code = this.makeFileAttrList(file);
		/* If some error occurred while making the attribute list then the corresponding error code is returned */
		if (code!=FileConstants.SUCCESS) {
			return code;
		}
		
		FileFilter fileFilter = new FileFilter();
		
		/* If the filter is not accepted then the result will be nothing */
		if (fileFilter.accept(this.attrList, filter)) {
			this.searchResult=this.attrList;
		}
		return FileConstants.SUCCESS;
	}
	
	
	/**
	 * Gives the complete list of files and directories under a directory.
	 * It can be made for one level under the directory or for all the subtree under it
	 * @param path -> Path of the file or directory
	 * @param filter -> Filter for the results. 
	 * 				   The filter must be a string following the next composition: attributeName filterString
	 * @param option -> Option code for the type of search (one level or sub-tree)
	 * @return -> An integer code indicating if the operation was done successfully or not.
	 * 			  In the case of error the corresponding error code is returned
	 */
	private int searchDeeper (String dirPath, String filter, int option) {
		
		File file = new File(dirPath);
		
		/* If the file does not exist then this is indicated by a error code */
		if (file.exists()==false) {	
			return FileConstants.FILE_NOT_FOUND;
		}
		/* If the file is not a directory then this is indicated by a error code */
		if (file.isDirectory()==false) {
			
			return FileConstants.WRONG_FILE_TYPE;
		}
		
		/* Creates the list of directories and files under a directory */
		File [] files = file.listFiles();
		
		/* For each one of the files and directories under the file ... */
		for (int i=0; files!=null && i<files.length; i++) {
			if (files[i]==null) continue;
			if (this.makeFileAttrList(files[i])!=FileConstants.SUCCESS) continue;
			FileFilter fileFilter = new FileFilter();
			/* If the filter is accepted ... */
			if (fileFilter.accept(this.attrList, filter)) {
				/* The file or directory is added as part of the result */
				this.searchResult.put(files[i].getPath(),this.attrList);
			}
			/* If the desired operation is one level search then the operation does not continue deeper */
			if (option==FileConstants.ONE_LEVEL_SEARCH) continue;
			/* It will be deeper if the current file is a directory */
			if (files[i].isDirectory()) {
				this.searchDeeper(files[i].getPath(), filter, option);
			}
		}
		
		return FileConstants.SUCCESS;
	}
	
	
	/**
	 * Creates a set of file or directory attributes
	 * @param file -> The file where to get the attributes from
	 * @return -> An integer code indicating if the operation was done successfully or not.
	 * 			  In the case of error the corresponding error code is returned
	 */
	private int makeFileAttrList (File file) {
		
		try {
			HashMap attrList = new HashMap();
			
			/* Creates the attribute list */
			if (file.isDirectory()) {
				//attrList.put("type", "directory");
				//attrList.put("content", "");
				attrList.put("objectclass", "directory");
				attrList.put("dir", file.getName());
			}
			else {
				//attrList.put("type", "file");
				
				RandomAccessFile fileRA = new RandomAccessFile(file.getAbsolutePath(),"rw");
				FileBasicOperations operations = new FileBasicOperations(fileRA);
				attrList.put("content", operations.getContent((int)file.length()));
				
				attrList.put("objectclass", "file");
				fileRA.close();
				attrList.put("file", file.getName());
			}
			attrList.put("hidden", file.isHidden()?"yes":"no");
			//TODO First you must implement "Refresh attributes" //attrList.put("size", (file.isFile())?(new Integer((int)file.length())).toString():"-1");
			
			attrList.put("permits",(file.canRead()?"r":"")+/*(file.canExecute()?"x":"")+*/(file.canWrite()?"w":""));
			attrList.put("path",file.getPath());
			
			/* If everything was realized successfully then this is the new result for search */
			this.attrList = attrList;
			
			return FileConstants.SUCCESS;
		}
		catch (FileNotFoundException fne) {
			
			return FileConstants.FILE_NOT_FOUND;
		}
		catch (IOException ioe) {
			
			return FileConstants.IO_ERROR;
		}
	}
}
